//===============================================================================
// Microsoft patterns & practices Enterprise Library
// Configuration Application Block
//===============================================================================
// Copyright  Microsoft Corporation.  All rights reserved.
// THIS CODE AND INFORMATION IS PROVIDED "AS IS" WITHOUT WARRANTY
// OF ANY KIND, EITHER EXPRESSED OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
// FITNESS FOR A PARTICULAR PURPOSE.
//===============================================================================

using System;
using System.Collections.Generic;
using Microsoft.VisualStudio.QualityTools.UnitTesting.Framework;

namespace Microsoft.Practices.EnterpriseLibrary.Configuration.Injection.Tests
{
	[TestClass]
	public class NamedObjectInjectionFactoryFixture
	{

		private const string SectionName = "namedObjects";

		[TestMethod]
		public void CreateTheDefaultNamedInstance()
		{
			TestFactory factory = new TestFactory();
			NamedObjectToCreate obj = factory.CreateDefault(new TestConfigurationSource());

			Assert.IsNotNull(obj);
			Assert.AreEqual<string>("fred", obj.Name);
		}

		[TestMethod]
		public void CreateANamedInstance()
		{
			TestFactory factory = new TestFactory();
            NamedObjectToCreate obj = factory.Create("willma", new TestConfigurationSource());

			Assert.IsNotNull(obj);
			Assert.AreEqual<string>("willma", obj.Name);
		}

		[TestMethod]
		[ExpectedException(typeof(InvalidOperationException))]
		public void TryToCreateAnInstanceThatDoesNotExist()
		{
			TestFactory factory = new TestFactory();
            factory.Create("boo", new TestConfigurationSource());
		}

		class TestConfigurationSource : IConfigurationSource
		{
			public object GetSection(string sectionName)
			{
				NamedObjectSection section = new NamedObjectSection();
				section.DefaultInstance = "fred";
				section.NamedObjects.Add("fred", new NamedObjectConfiguration("fred"));
				section.NamedObjects.Add("willma", new NamedObjectConfiguration("willma"));
				section.NamedObjects.Add("betty", new NamedObjectConfiguration("betty"));
				section.NamedObjects.Add("barny", new NamedObjectConfiguration("barney"));
				return section;
			}
		}

		class TestFactory : NamedObjectInjectionFactory<NamedObjectToCreate, NamedObjectConfiguration>
		{
			protected override NamedObjectConfiguration GetNamedConfiguration(string name, IConfigurationSource configurationSource)
			{
				NamedObjectSection section = (NamedObjectSection)configurationSource.GetSection(SectionName);
				return section.NamedObjects[name];
			}

			protected override string GetDefaultInstanceName(IConfigurationSource configurationSource)
			{
				NamedObjectSection section = (NamedObjectSection)configurationSource.GetSection(SectionName);
				return section.DefaultInstance;
			}
		}

		class NamedObjectSection
		{
			private Dictionary<string, NamedObjectConfiguration> namedObjects;
			private string defaultInstance;

			public NamedObjectSection()
			{
				namedObjects = new Dictionary<string, NamedObjectConfiguration>();
			}

			public string DefaultInstance
			{
				get { return defaultInstance; }
				set { defaultInstance = value; }
			}

			public Dictionary<string, NamedObjectConfiguration> NamedObjects
			{
				get { return namedObjects; }
			}
		}

		class NamedObjectToCreate
		{
			private readonly string name;

			public NamedObjectToCreate(string name)
			{
				this.name = name;
			}

			public string Name
			{
				get { return name; }
			}
		}

		[PropertyParameterAttribute(0, typeof(string), "Name")]
		class NamedObjectConfiguration
		{
			private readonly string name;

			public NamedObjectConfiguration(string name)
			{
				this.name = name;
			}

			public string Name
			{
				get { return name; }
			}
		}		
	}
}
